
-- EngineScript.lua für RSSLO 5022 / Desiro Classic -- Realistische Fahrphysik, Sifa, PZB, Türsperre, Startbedingungen
local engineOn = false
local starting = false
local mainSwitch = false
local batteriesOn = false
local reverserSet = false
local throttle = 0
local brake = 0
local speed = 0 -- [m/s]
local rpm = 0
local targetRpm = 0
local startTimer = 0

-- Fahrzeugdaten
local leergewicht = 70300 -- kg (70,3 t)
local maxLeistung = 670000 -- W (2x335 kW)
local maxZugkraft = 120000 -- N
local maxSpeed = 33.33 -- m/s (120 km/h)
local rollwiderstand = 2 -- N/t
local cw = 1.5
local frontalfläche = 7.9
local luftdichte = 1.2

-- Türen
local doorsClosed = false

-- Sifa
local sifaTimer = 0
local sifaMaxTime = 30 -- Sekunden bis Warnung
local sifaActive = true
local sifaWarn = false
local sifaBrake = false

-- PZB
local pzbActive = false
local pzbRestriktiv = false

function Initialise()
    engineOn = false
    starting = false
    mainSwitch = false
    batteriesOn = false
    reverserSet = false
    throttle = 0
    brake = 0
    speed = 0
    rpm = 0
    targetRpm = 0
    startTimer = 0
    doorsClosed = false
    sifaTimer = 0
    sifaWarn = false
    sifaBrake = false
    pzbActive = false
    pzbRestriktiv = false

    Call("SetControlValue", "EngineRPM", 0, 0)
    Call("StopSound", "engine_idle.wav")
    Call("StopSound", "engine_run.wav")
end

function OnControlValueChange(name, index, value)
    if name == "BatteryOn" and value == 1 then batteriesOn = true end
    if name == "MainSwitch" and value == 1 and batteriesOn then mainSwitch = true end
    if name == "Reverser" and value ~= 0 then reverserSet = true end

    if name == "EngineStart" and value == 1 and not engineOn and mainSwitch and batteriesOn then
        starting = true
        startTimer = 0
        Call("PlaySound", "engine_start.wav")
    end

    if name == "Throttle" then throttle = value end
    if name == "Brake" then brake = value end
    if name == "DoorsClosed" then doorsClosed = (value == 1) end

    if name == "Sifa" and value == 1 then
        sifaTimer = 0
        sifaWarn = false
        sifaBrake = false
    end

    if name == "PZB" and value == 1 then pzbActive = not pzbActive end
    if name == "PZB_Restriktiv" and value == 1 and pzbActive then pzbRestriktiv = true end
    if name == "PZB_Frei" and value == 1 then pzbRestriktiv = false end
end

function Update(time)
    -- Sifa
    if sifaActive and engineOn and speed > 1 then
        sifaTimer = sifaTimer + time
        if sifaTimer > sifaMaxTime and not sifaWarn then
            sifaWarn = true
            Call("PlaySound", "sifa_warn.wav")
        end
        if sifaTimer > (sifaMaxTime + 5) and not sifaBrake then
            sifaBrake = true
            Call("PlaySound", "sifa_brake.wav")
            brake = 1
        end
    end

    -- Startlogik
    if starting then
        startTimer = startTimer + time
        if startTimer > 3 then
            engineOn = true
            starting = false
            rpm = 800
            targetRpm = rpm
            Call("PlayLoopedSound", "engine_idle.wav")
        end
        return
    end

    if not (engineOn and mainSwitch and batteriesOn and reverserSet) then
        rpm = 0
        Call("SetControlValue", "EngineRPM", 0, rpm)
        return
    end

    -- Türsperre
    local realThrottle = throttle
    if not doorsClosed then
        realThrottle = 0
        if speed < 1 then
            Call("PlaySound", "doors_open_warn.wav")
        end
    end

    -- PZB Restriktiv
    if pzbActive and pzbRestriktiv and speed > 45/3.6 then
        realThrottle = 0.2
    end

    -- Fahrphysik
    local aktuelleLeistung = realThrottle * maxLeistung
    local v = math.max(speed, 1)
    local zugkraftBegrenzt = math.min(aktuelleLeistung / v, maxZugkraft)
    local bremskraft = brake * maxZugkraft
    local f_roll = leergewicht * rollwiderstand * 0.001
    local f_luft = 0.5 * luftdichte * cw * frontalfläche * speed * speed
    local f_netto = zugkraftBegrenzt - f_roll - f_luft - bremskraft
    local beschleunigung = f_netto / leergewicht
    speed = speed + beschleunigung * time

    if speed < 0 then speed = 0 end
    if speed > maxSpeed then speed = maxSpeed end

    Call("SetControlValue", "Speedometer", 0, speed * 3.6)

    targetRpm = 800 + realThrottle * 1400 + math.min(speed / maxSpeed, 1) * 500
    rpm = rpm + (targetRpm - rpm) * 0.05
    Call("SetControlValue", "EngineRPM", 0, rpm)

    -- Sounds
    if realThrottle < 0.1 and speed < 0.5 then
        Call("PlayLoopedSound", "engine_idle.wav")
        Call("StopSound", "engine_run.wav")
    else
        Call("StopSound", "engine_idle.wav")
        local pitch = 1.0 + realThrottle * 0.4 + speed / maxSpeed * 0.2
        Call("PlayLoopedSound", "engine_run.wav", pitch)
    end
end
