--
-- FreePZB - Faithful implementation in Lua of the German train protection
-- system PZB90 for use in train simulation games
--
-- Copyright (c) 2012 Andreas Bombe <aeb@debian.org>
-- 
-- Permission is hereby granted, free of charge, to any person obtaining a copy
-- of this software and associated documentation files (the "Software"), to
-- deal in the Software without restriction, including without limitation the
-- rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
-- sell copies of the Software, and to permit persons to whom the Software is
-- furnished to do so, subject to the following conditions:
-- 
-- The above copyright notice and this permission notice shall be included in
-- all copies or substantial portions of the Software.
-- 
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
-- FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
-- AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
-- LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
-- FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
-- IN THE SOFTWARE.


local P = {}

if _REQUIREDNAME == nil then
	freepzb = P
else
	_G[_REQUIREDNAME] = P
end
setfenv(1, P)


BLAU_AUS      = 0
BLAU_55       = 1
BLAU_70       = 2
BLAU_85       = 3
BLAU_BLINK_55 = 4
BLAU_BLINK_70 = 5
BLAU_BLINK_85 = 6
BLAU_WECHSEL  = 7

MAGNET_AUS    = 0
MAGNET_1000   = 1
MAGNET_500    = 2
MAGNET_ZWANG  = 3
MAGNET_STOER  = 4


local parameter_O = {
	blau_frei          = BLAU_85,
	blau_1000hz        = BLAU_BLINK_85,
	pruef_hoechst      = 165,
	pruef_1000hz       = 85,
	ablauf_1000hz      = 23,
	pruef_500hz        = 45,
	pruef_500hz_start  = 65,
	pruef_r500hz_start = 45,
}

local parameter_M = {
	blau_frei          = BLAU_70,
	blau_1000hz        = BLAU_BLINK_70,
	pruef_hoechst      = 125,
	pruef_1000hz       = 70,
	ablauf_1000hz      = 29,
	pruef_500hz        = 35,
	pruef_500hz_start  = 50,
	pruef_r500hz_start = 25,
}

local parameter_U = {
	blau_frei          = BLAU_55,
	blau_1000hz        = BLAU_BLINK_55,
	pruef_hoechst      = 105,
	pruef_1000hz       = 55,
	ablauf_1000hz      = 38,
	pruef_500hz        = 25,
	pruef_500hz_start  = 40,
	pruef_r500hz_start = 25,
}


function create(simzeit, simstrecke, zugart)
	local inst = {}

	if simzeit < 0 or simstrecke < 0 then
		return nil
	end

	inst.zeit = simzeit
	inst.strecke = simstrecke

	inst.zwangsbremsung = false
	inst.zwangsbremsung_grund = nil
	inst.ueberwbremsung = nil
	inst.aktive_pruef = 999
	inst.aktive_umschalt = 0

	inst.letzter_magnet = 0
	inst.letzte_wachtaste = false
	inst.wach_faellig = -1
	inst.beginn_unter_umschalt = -1

	inst.start_1000hz = -1000
	inst.ende_1000hz = -1000
	inst.ende_r1000hz = -1000
	inst.ende_1000hz_sperre = -1000
	inst.ende_ablauf_1000hz = -1
	inst.start_500hz = -1000
	inst.ende_500hz = -1000
	inst.restrikt_500hz = false
	inst.ende_start = -1000

	inst.befreit = -1000
	inst.unterdrueck_lampe1000 = -1
	inst.akust_signal_ende = -1
	inst.letzte_restriktiv = false

	if zugart == "O" then
		inst.parameter = parameter_O
	elseif zugart == "M" then
		inst.parameter = parameter_M
	elseif zugart == "U" then
		inst.parameter = parameter_U
	else
		return nil
	end
	inst.zugart = zugart

	inst.lampen_blau = inst.parameter.blau_frei
	inst.lampe_befehl = false
	inst.lampen_magnet = MAGNET_AUS
	inst.akust_signal = false

	return inst
end


function activate_start(inst, pos)
	inst.ende_start = pos + 550
end



function update(inst, zeit, pos, geschw, magnet_aktiv, tasten)
	local aktive_pruef = 999
	local aktive_pruef_name = ""
	local blau = inst.parameter.blau_frei
	local lmag = MAGNET_AUS
	local umschalt_geschw = 0
	local m2000 = false
	local aktiviere_1000 = false
	local akust_signal = false
	local restriktiv = false

	-- Umschaltgeschwindigkeit
	if (inst.ende_1000hz > pos and inst.befreit < pos) or inst.ende_500hz > pos then
		umschalt_geschw = 10
	end

	if inst.start_500hz + 153 > pos and inst.zugart == "O" then
		umschalt_geschw = (inst.start_500hz + 153 - pos) / 153 * 20 + 10
	end

	if geschw < umschalt_geschw then
		if inst.beginn_unter_umschalt < 0 then
			inst.beginn_unter_umschalt = zeit
		elseif zeit - inst.beginn_unter_umschalt >= 15 then
			if inst.ende_1000hz > inst.ende_r1000hz then
				inst.ende_r1000hz = inst.ende_1000hz
			end

			if inst.ende_500hz > pos and not inst.restrikt_500hz then
				inst.restrikt_500hz = true
				if pos < inst.start_500hz + 100 then
					inst.ende_500hz = inst.start_500hz + 200
				end
			end
		end
	else
		inst.beginn_unter_umschalt = -1
	end

	-- Behandlung Beeinflussungen
	if magnet_aktiv ~= inst.letzter_magnet then
		if inst.letzter_magnet == 1000 then
			inst.start_1000hz = pos
			inst.wach_faellig = zeit + 4
		elseif inst.letzter_magnet == 500 then
			if inst.ende_r1000hz < pos then
				inst.ende_500hz = pos + 250
				inst.restrikt_500hz = false
			else
				inst.ende_500hz = pos + 200
				inst.restrikt_500hz = true
			end
			inst.start_500hz = pos

			if inst.befreit > pos then
				inst.ueberwbremsung = "500 Hz w. befreit"
				inst.befreit = -1000
			end
		elseif inst.letzter_magnet == 2000 then
			m2000 = true
			-- weitere Behandlung bei Befehlstaste
		end
		inst.letzter_magnet = magnet_aktiv
	end

	-- Wachsam
	if tasten.wachsam then
		akust_signal = true
	elseif inst.letzte_wachtaste and inst.wach_faellig > 0 then
		aktiviere_1000 = true
		inst.wach_faellig = -1
	end
	inst.letzte_wachtaste = tasten.wachsam

	if inst.wach_faellig > 0 and inst.wach_faellig < zeit then
		inst.ueberwbremsung = "nicht wachsam"
		aktiviere_1000 = true
		inst.wach_faellig = -1
	end

	if aktiviere_1000 then
		if inst.ende_1000hz < pos and inst.ende_r1000hz < pos then
			inst.ende_ablauf_1000hz = zeit + inst.parameter.ablauf_1000hz
		end
		if inst.ende_1000hz_sperre > pos then
			inst.unterdrueck_lampe1000 = zeit + 0.5
		end
		inst.ende_1000hz = inst.start_1000hz + 1250
		inst.ende_1000hz_sperre = inst.start_1000hz + 700
		inst.befreit = -1000
	end

	-- Startprogramm
	if inst.ende_start > pos and geschw >= 5 then
		if inst.ende_start > inst.ende_r1000hz then
			inst.ende_r1000hz = inst.ende_start
		end
		inst.ende_start = -1000
	end

	-- Befreiung
	if tasten.frei then
		akust_signal = true

		if inst.ende_1000hz_sperre < pos and inst.ende_500hz < pos then
			if inst.ende_1000hz > inst.ende_r1000hz then
				inst.befreit = inst.ende_1000hz
			else
				inst.befreit = inst.ende_r1000hz
			end
		end

		if geschw < 5 then
			inst.ueberwbremsung = nil
		end
	end

	-- LM 1000 Hz
	if inst.ende_1000hz_sperre > pos and inst.unterdrueck_lampe1000 < zeit then
		lmag = MAGNET_1000
	end

	-- Ueberwachung 1000 Hz
	if inst.ende_1000hz > pos and inst.befreit < pos then
		blau = inst.parameter.blau_1000hz
		aktive_pruef = inst.parameter.pruef_1000hz
		aktive_pruef_name = "1000 Hz"

		if inst.ende_ablauf_1000hz > zeit then
			aktive_pruef = aktive_pruef +
				(inst.ende_ablauf_1000hz - zeit) / inst.parameter.ablauf_1000hz *
				(inst.parameter.pruef_hoechst - inst.parameter.pruef_1000hz)
		end
	end

	-- Ueberwachung 1000 Hz restriktiv
	if inst.ende_r1000hz > pos and inst.befreit < pos then
		blau = BLAU_WECHSEL
		aktive_pruef = 45
		aktive_pruef_name = "1000 Hz restriktiv"
		restriktiv = true
	end

	-- Uberwachung 500 Hz
	if inst.ende_500hz > pos then
		local pruef = inst.parameter.pruef_500hz
		local pruef_start = inst.parameter.pruef_500hz_start
		aktive_pruef_name = "500 Hz"

		if inst.restrikt_500hz then
			blau = BLAU_WECHSEL
			pruef = 25
			pruef_start = inst.parameter.pruef_r500hz_start
			aktive_pruef_name = "500 Hz restriktiv"
			restriktiv = true
		end

		lmag = MAGNET_500
		aktive_pruef = pruef

		if pos < inst.start_500hz + 153 then
			aktive_pruef = aktive_pruef +
				(inst.start_500hz + 153 - pos) / 153 * (pruef_start - pruef)
		end
	end

	-- Befehlstaste und 2000 Hz
	if not tasten.befehl then
		inst.lampe_befehl = false
		if m2000 then
			inst.ueberwbremsung = "2000 Hz"
		end
	else
		if m2000 then
			inst.lampe_befehl = true
		end

		if inst.lampe_befehl == true then
			if 40 < aktive_pruef then
				aktive_pruef = 40
				aktive_pruef_name = "Befehl 40"
			end
			akust_signal = true
		end
	end

	-- Akustisches Signal fuer Ende restriktiv
	if not restriktiv and inst.letzte_restriktiv then
		inst.akust_signal_ende = zeit + 0.5
	end
	inst.letzte_restriktiv = restriktiv

	if inst.akust_signal_ende > zeit then
		akust_signal = true
	end

	-- Test Uberschreitung der Pruefgeschwindigkeit
	if not inst.ueberwbremsung and geschw >= aktive_pruef + 1 then
		inst.ueberwbremsung = aktive_pruef_name
	end


	if inst.ueberwbremsung then
		inst.zwangsbremsung = true
		inst.zwangsbremsung_grund = inst.ueberwbremsung
		blau = BLAU_AUS
		lmag = MAGNET_ZWANG
		akust_signal = true
	else
		inst.zwangsbremsung = false
	end

	inst.aktive_pruef = aktive_pruef
	inst.aktive_umschalt = umschalt_geschw
	inst.lampen_blau = blau
	inst.lampen_magnet = lmag
	inst.akust_signal = akust_signal
end


return P
